<?php

namespace App\Http\Controllers;

use Illuminate\Http\Request;
use App\Models\User;
use App\Models\SchoolClass;
use App\Models\PaymentRequest;
use App\Models\Result;
use App\Models\SchoolSession;
use App\Models\Subject;
use App\Models\Wallet;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Cache;

class DashboardController extends Controller
{
    public function index()
    {
        $user = auth()->user();

        // Redirect students to their dashboard
        if ($user->isStudent()) {
            return redirect()->route('student.dashboard');
        }

        // Get counts based on user role
        $data = [
            'students_count' => $this->getStudentsCount($user),
            'staff_count' => $this->getStaffCount($user),
            'classes_count' => $this->getClassesCount($user),
            'wallet_balance' => $this->getWalletBalance($user),
            'current_session' => SchoolSession::where('is_active', true)->first(),
            'assigned_classes' => $this->getAssignedClasses($user),
            'teacher_subjects' => $this->getTeacherSubjects($user),
            'pending_results' => $this->getPendingResults($user),
        ];

        return view('dashboard', $data);
    }

    private function getStudentsCount($user)
    {
        if ($user->isStudent()) {
            return 1;
        }

        if ($user->isTeacher()) {
            $cacheKey = "teacher_students_count_{$user->id}";
            return Cache::remember($cacheKey, 3600, function () use ($user) {
                $teacherClassIds = $user->classes()->pluck('school_classes.id');
                return User::whereHas('batch', function ($query) use ($teacherClassIds) {
                    $query->whereIn('school_class_id', $teacherClassIds);
                })->whereDoesntHave('roles')->count();
            });
        }

        if ($user->isAdmin() || $user->isHeadTeacher() || $user->isBursar()) {
            return Cache::remember('total_students_count', 3600, function () {
                return User::whereDoesntHave('roles')->count();
            });
        }

        return 0;
    }

    private function getStaffCount($user)
    {
        if ($user->isStudent() || $user->isTeacher()) {
            return 0;
        }

        return Cache::remember('total_staff_count', 3600, function () {
            return User::whereHas('roles')->count();
        });
    }

    private function getClassesCount($user)
    {
        if ($user->isStudent()) {
            return 1;
        }

        if ($user->isTeacher()) {
            $cacheKey = "teacher_classes_count_{$user->id}";
            return Cache::remember($cacheKey, 3600, function () use ($user) {
                return $user->classes()->count();
            });
        }

        if ($user->isAdmin() || $user->isHeadTeacher() || $user->isBursar()) {
            return Cache::remember('total_classes_count', 3600, function () {
                return SchoolClass::where('is_active', true)->count();
            });
        }

        return 0;
    }

    private function getTotalPayments($user)
    {
        if ($user->isStudent()) {
            return PaymentRequest::where('student_id', $user->id)
                ->where('status', 'paid')
                ->sum('amount_paid');
        }

        if ($user->isTeacher()) {
            return 0; // Teachers don't handle payments
        }

        return PaymentRequest::where('status', 'paid')->sum('amount_paid');
    }

    private function getPendingPayments($user)
    {
        if ($user->isStudent()) {
            return PaymentRequest::where('student_id', $user->id)
                ->whereIn('status', ['unpaid', 'processing', 'partial_payment'])
                ->count();
        }

        if ($user->isTeacher()) {
            return 0;
        }

        return PaymentRequest::whereIn('status', ['unpaid', 'processing', 'partial_payment'])->count();
    }

    private function getAssignedClasses($user)
    {
        if ($user->isTeacher() || $user->isHeadTeacher()) {
            return $user->classes()->where('is_active', true)->get();
        }

        return collect();
    }

    private function getTeacherSubjects($user)
    {
        if ($user->isTeacher()) {
            return $user->classes()->where('is_active', true)
                ->with('subjects')
                ->get()
                ->pluck('subjects')
                ->flatten()
                ->unique('id')
                ->count();
        }

        return 0;
    }

    private function getPendingResults($user)
    {
        if ($user->isTeacher()) {
            $currentSession = SchoolSession::where('is_active', true)->first();
            if (!$currentSession) return 0;

            $teacherClassIds = $user->classes()->pluck('school_classes.id');
            return User::whereHas('batch', function ($query) use ($teacherClassIds) {
                $query->whereIn('school_class_id', $teacherClassIds);
            })
                ->whereDoesntHave('results', function ($query) use ($currentSession) {
                    $query->where('session_id', $currentSession->id);
                })
                ->whereDoesntHave('roles')
                ->count();
        }

        return 0;
    }

    private function getWalletBalance($user)
    {
        if ($user->isStudent()) {
            // For students, show their individual wallet balance if they have one
            return 0; // Students don't have individual wallets in this system
        }

        if ($user->isTeacher()) {
            return 0; // Teachers don't handle wallet balances
        }

        // For admin, head teacher, and bursar - show total of all wallet balances
        return (int)Wallet::get()->sum('balance');
    }
}
