<?php

namespace App\Http\Controllers;

use App\Services\PayrollService;
use App\Traits\HasNotifications;

class PayrollController extends Controller
{
    use HasNotifications;

    protected $payrollService;

    public function __construct(PayrollService $payrollService)
    {
        $this->payrollService = $payrollService;
    }

    public function index()
    {
        return view('payroll.index');
    }

    public function salaryHistory()
    {
        return view('payroll.salary-history');
    }

    public function staffSalaries()
    {
        return view('payroll.staff-salaries');
    }

    public function mySalaryHistory()
    {
        return view('staff.salary-history');
    }

    public function downloadPayroll($month, $year, $format = 'csv')
    {
        $payrollData = $this->payrollService->getPayrollData($month, $year);
        
        if ($payrollData->isEmpty()) {
            session()->flash('toast', [
                'type' => 'error',
                'message' => 'No payroll data found for the selected period!'
            ]);
            return redirect()->back();
        }

        if ($format === 'excel') {
            return $this->downloadExcel($payrollData, $month, $year);
        }

        return $this->downloadCsv($payrollData, $month, $year);
    }

    private function downloadCsv($payrollData, $month, $year)
    {
        $filename = "payroll_{$month}_{$year}.csv";
        
        $headers = [
            'Content-Type' => 'text/csv',
            'Content-Disposition' => "attachment; filename={$filename}",
        ];

        $callback = function() use ($payrollData) {
            $file = fopen('php://output', 'w');
            
            fputcsv($file, [
                'Staff Name',
                'Email',
                'Month',
                'Year',
                'Base Salary',
                'Total Bonus',
                'Total Deductions',
                'Net Salary',
                'Status',
                'Paid Date'
            ]);

            foreach ($payrollData as $salary) {
                fputcsv($file, [
                    $salary->user->name,
                    $salary->user->email,
                    $salary->month,
                    $salary->year,
                    number_format($salary->monthly_salary, 2),
                    number_format($salary->total_bonus, 2),
                    number_format($salary->total_deductions, 2),
                    number_format($salary->net_salary, 2),
                    ucfirst($salary->status),
                    $salary->paid_at ? $salary->paid_at->format('Y-m-d H:i:s') : ''
                ]);
            }

            fclose($file);
        };

        return response()->stream($callback, 200, $headers);
    }

    private function downloadExcel($payrollData, $month, $year)
    {
        $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
        $sheet = $spreadsheet->getActiveSheet();
        $sheet->setTitle('Payroll');
        
        // Headers
        $headers = ['Staff Name', 'Email', 'Month', 'Year', 'Base Salary', 'Total Bonus', 'Total Deductions', 'Net Salary', 'Status', 'Paid Date'];
        $sheet->fromArray($headers, null, 'A1');
        
        // Data
        $row = 2;
        foreach ($payrollData as $salary) {
            $sheet->setCellValue('A' . $row, $salary->user->name);
            $sheet->setCellValue('B' . $row, $salary->user->email);
            $sheet->setCellValue('C' . $row, $salary->month);
            $sheet->setCellValue('D' . $row, $salary->year);
            $sheet->setCellValue('E' . $row, $salary->monthly_salary);
            $sheet->setCellValue('F' . $row, $salary->total_bonus);
            $sheet->setCellValue('G' . $row, $salary->total_deductions);
            $sheet->setCellValue('H' . $row, $salary->net_salary);
            $sheet->setCellValue('I' . $row, ucfirst($salary->status));
            $sheet->setCellValue('J' . $row, $salary->paid_at ? $salary->paid_at->format('Y-m-d H:i:s') : '');
            $row++;
        }
        
        // Format currency columns
        $sheet->getStyle('E:E')->getNumberFormat()->setFormatCode('"₦"#,##0.00');
        $sheet->getStyle('F:F')->getNumberFormat()->setFormatCode('"₦"#,##0.00');
        $sheet->getStyle('G:G')->getNumberFormat()->setFormatCode('"₦"#,##0.00');
        $sheet->getStyle('H:H')->getNumberFormat()->setFormatCode('"₦"#,##0.00');
        
        // Auto-size columns
        foreach (range('A', 'J') as $col) {
            $sheet->getColumnDimension($col)->setAutoSize(true);
        }
        
        $filename = "payroll_{$month}_{$year}.xlsx";
        
        $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);
        
        return response()->streamDownload(function() use ($writer) {
            $writer->save('php://output');
        }, $filename, [
            'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
        ]);
    }

    public function printPayroll($month, $year)
    {
        $payrollData = $this->payrollService->getPayrollData($month, $year);
        
        if ($payrollData->isEmpty()) {
            session()->flash('toast', [
                'type' => 'error',
                'message' => 'No payroll data found for the selected period!'
            ]);
            return redirect()->back();
        }

        return view('payroll.print-payroll', compact('payrollData', 'month', 'year'));
    }

    public function getAdjustments($salaryHistoryId)
    {
        $salaryHistory = \App\Models\SalaryHistory::with(['adjustments', 'user'])->findOrFail($salaryHistoryId);
        
        return view('payroll.adjustments-list', compact('salaryHistory'));
    }
}
