<?php

namespace App\Http\Controllers\Student;

use App\Http\Controllers\Controller;
use App\Traits\HasNotifications;
use App\Services\ViewResultsService;
use App\Services\AnnualResultService;
use App\Services\QrCodeService;
use App\Models\SchoolSession;
use App\Models\ResultApproval;


class StudentResultController extends Controller
{
    use HasNotifications;

    public function termlyResults()
    {
        return view('student.results.termly');
    }

    public function annualResults()
    {
        return view('student.results.annual');
    }

    public function viewTermlyResult(SchoolSession $session)
    {
        try {
            $student = auth()->user();
            
            // Check if student has approved result for this session
            $hasApprovedResult = ResultApproval::where('student_id', $student->id)
                ->where('session_id', $session->id)
                ->exists();
            
            if (!$hasApprovedResult) {
                abort(403, 'You do not have an approved result for this session.');
            }
            
            $viewResultsService = new ViewResultsService();
            $data = $viewResultsService->getStudentCompleteResult($student->id, $session->id);
            
            if (!$data) {
                abort(404, 'Student result not found');
            }
            
            $qrCode = null;
            if ($data['approval']) {
                $qrCodeService = new QrCodeService();
                $currentUrl = request()->url();
                $qrCode = $qrCodeService->generateResultQrCode($data['student'], $data['approval'], $data['session'], $currentUrl);
            }
            
            return view('student.results.view-termly', compact('data', 'qrCode'));
        } catch (\Exception $e) {
            abort(500, $e->getMessage());
        }
    }

    public function printTermlyResult(SchoolSession $session)
    {
        try {
            $student = auth()->user();
            
            // Check if student has approved result for this session
            $hasApprovedResult = ResultApproval::where('student_id', $student->id)
                ->where('session_id', $session->id)
                ->exists();
            
            if (!$hasApprovedResult) {
                abort(403, 'You do not have an approved result for this session.');
            }
            
            $viewResultsService = new ViewResultsService();
            $data = $viewResultsService->getStudentCompleteResult($student->id, $session->id);
            
            if (!$data) {
                abort(404, 'Student result not found');
            }
            
            $qrCode = null;
            if ($data['approval']) {
                $qrCodeService = new QrCodeService();
                $currentUrl = request()->url();
                $qrCode = $qrCodeService->generateResultQrCode($data['student'], $data['approval'], $data['session'], $currentUrl);
            }
            
            return view('results.student-result', compact('data', 'qrCode'));
        } catch (\Exception $e) {
            abort(500, $e->getMessage());
        }
    }

    public function viewAnnualResult($year)
    {
        $student = auth()->user();
        
        // Check if student has approved annual result for this year
        $hasApprovedResult = \App\Models\AnnualResultApproval::where('student_id', $student->id)
            ->where('year', $year)
            ->exists();
        
        if (!$hasApprovedResult) {
            abort(403, 'You do not have an approved annual result for this year.');
        }
        
        $annualResultService = new AnnualResultService();
        $data = $annualResultService->getStudentAnnualResultForPrint($student->id, $year);
        
        if (!$data) {
            abort(404, 'Annual result not found');
        }
        
        $qrCode = null;
        if ($data['approval']) {
            $qrCodeService = new QrCodeService();
            $currentUrl = request()->url();
            $qrCode = $qrCodeService->generateAnnualResultQrCode($data['student'], $data['year'], $currentUrl);
        }
        
        return view('student.results.view-annual', compact('data', 'qrCode'));
    }

    public function printAnnualResult($year)
    {
        $student = auth()->user();
        
        // Check if student has approved annual result for this year
        $hasApprovedResult = \App\Models\AnnualResultApproval::where('student_id', $student->id)
            ->where('year', $year)
            ->exists();
        
        if (!$hasApprovedResult) {
            abort(403, 'You do not have an approved annual result for this year.');
        }
        
        $annualResultService = new AnnualResultService();
        $data = $annualResultService->getStudentAnnualResultForPrint($student->id, $year);
        
        
        if (!$data) {
            abort(404, 'Annual result not found');
        }
        
        $qrCode = null;
        if ($data['approval']) {
            $qrCodeService = new QrCodeService();
            $currentUrl = request()->url();
            $qrCode = $qrCodeService->generateAnnualResultQrCode($data['student'], $data['year'], $currentUrl);
        }
        
        return view('results.print-student-annual-result', compact('data', 'qrCode'));
    }


}