<?php

namespace App\Livewire;

use Livewire\Component;
use Livewire\WithPagination;
use App\Services\PaymentService;
use App\Models\PaymentRequest;
use App\Traits\HasNotifications;

class PaymentRequestsTable extends Component
{
    use WithPagination, HasNotifications;

    public $search = '';
    public $statusFilter = '';
    public $perPage = 10;
    public $selectedPayment = null;
    public $showUpdateModal = false;
    public $showEditModal = false;
    public $newStatus = '';
    public $paymentAmount = '';
    public $paymentNotes = '';
    
    // Edit form properties
    public $editTitle = '';
    public $editDescription = '';
    public $editAmount = '';
    public $editCurrency = '';

    protected $paginationTheme = 'bootstrap';
    
    public function paginationView()
    {
        return 'custom.pagination';
    }

    public function updatingSearch()
    {
        $this->resetPage();
    }

    public function updatingStatusFilter()
    {
        $this->resetPage();
    }
    
    public function updatedPerPage()
    {
        $this->resetPage();
    }

    public function openUpdateModal($paymentId)
    {
        $this->selectedPayment = PaymentRequest::find($paymentId);
        if (in_array($this->selectedPayment->status, ['settled', 'cancelled'])) {
            session()->flash('toast', [
                'type' => 'error',
                'message' => 'Cannot update status of settled or cancelled payment requests'
            ]);
            return;
        }
        $this->newStatus = $this->selectedPayment->status;
        $this->showUpdateModal = true;
    }
    
    public function openEditModal($paymentId)
    {
        $this->selectedPayment = PaymentRequest::find($paymentId);
        if (in_array($this->selectedPayment->status, ['settled', 'cancelled'])) {
            session()->flash('toast', [
                'type' => 'error',
                'message' => 'Cannot edit settled or cancelled payment requests'
            ]);
            return;
        }
        $this->editTitle = $this->selectedPayment->title;
        $this->editDescription = $this->selectedPayment->description;
        $this->editAmount = $this->selectedPayment->amount;
        $this->editCurrency = $this->selectedPayment->currency;
        $this->showEditModal = true;
    }

    public function closeUpdateModal()
    {
        $this->showUpdateModal = false;
        $this->selectedPayment = null;
        $this->newStatus = '';
    }
    
    public function closeEditModal()
    {
        $this->showEditModal = false;
        $this->selectedPayment = null;
        $this->editTitle = '';
        $this->editDescription = '';
        $this->editAmount = '';
        $this->editCurrency = '';
    }

    public function updatePaymentStatus()
    {
        $this->validate(['newStatus' => 'required|in:active,cancelled,settled']);

        try {
            $this->selectedPayment->update(['status' => $this->newStatus]);
            
            session()->flash('toast', [
                'type' => 'success',
                'message' => 'Payment request status updated successfully!'
            ]);
            $this->closeUpdateModal();
        } catch (\Exception $e) {
            session()->flash('toast', [
                'type' => 'error',
                'message' => 'Error updating payment request status: ' . $e->getMessage()
            ]);
        }
    }
    
    public function updatePaymentRequest()
    {
        $this->validate([
            'editTitle' => 'required|string|max:255',
            'editDescription' => 'required|string',
            'editAmount' => 'required|numeric|min:0.01',
            'editCurrency' => 'required|string|size:3',
        ]);

        try {
            $this->selectedPayment->update([
                'title' => $this->editTitle,
                'description' => $this->editDescription,
                'amount' => $this->editAmount,
                'currency' => $this->editCurrency,
                'balance' => $this->editAmount - $this->selectedPayment->amount_paid,
            ]);
            
            session()->flash('toast', [
                'type' => 'success',
                'message' => 'Payment request updated successfully!'
            ]);
            $this->closeEditModal();
        } catch (\Exception $e) {
            session()->flash('toast', [
                'type' => 'error',
                'message' => 'Error updating payment request: ' . $e->getMessage()
            ]);
        }
    }

    public function render()
    {
        $query = PaymentRequest::with(['student', 'schoolClass', 'batch', 'createdBy', 'payments.recordedBy'])
            ->when($this->search, function ($q) {
                $q->where('title', 'like', '%' . $this->search . '%')
                  ->orWhere('description', 'like', '%' . $this->search . '%')
                  ->orWhereHas('student', function ($sq) {
                      $sq->where('first_name', 'like', '%' . $this->search . '%')
                         ->orWhere('last_name', 'like', '%' . $this->search . '%')
                         ->orWhere('other_names', 'like', '%' . $this->search . '%');
                  })
                  ->orWhereHas('schoolClass', function ($cq) {
                      $cq->where('name', 'like', '%' . $this->search . '%');
                  })
                  ->orWhereHas('batch', function ($bq) {
                      $bq->where('name', 'like', '%' . $this->search . '%');
                  });
            })
            ->when($this->statusFilter, function ($q) {
                $q->where('status', $this->statusFilter);
            })
            ->orderBy('created_at', 'desc');

        $payments = $query->paginate($this->perPage);

        return view('livewire.payment-requests-table', compact('payments'));
    }
}