<?php

namespace App\Livewire\Results;

use Livewire\Component;
use App\Models\SchoolSession;
use App\Models\AnnualResultApproval;
use App\Models\User;
use App\Models\SchoolClass;
use App\Services\AnnualResultService;
use Illuminate\Support\Facades\Auth;

class ViewAnnualResults extends Component
{
    public $selectedYear = '';
    public $classId = '';
    public $batchId = '';
    public $studentSearch = '';
    public $selectedStudent = null;
    public $filteredStudentId = null;
    public $years = [];
    public $classes = [];
    public $batches = [];
    public $approvedResults = [];
    public $isLoaded = false;
    public $selectedStudentData = null;
    public $showingResultFor = null;
    
    protected $annualResultService;
    
    public function boot(AnnualResultService $annualResultService)
    {
        $this->annualResultService = $annualResultService;
    }

    public function mount()
    {
        $this->years = SchoolSession::distinct()
            ->orderBy('start_year', 'desc')
            ->pluck('start_year')
            ->toArray();
            
        // Auto-select currently active session year
        $activeSession = \App\Models\SchoolSession::where('is_active', true)->first();
        if ($activeSession && in_array($activeSession->start_year, $this->years)) {
            $this->selectedYear = $activeSession->start_year;
        } elseif (!empty($this->years)) {
            $this->selectedYear = $this->years[0]; // Select the most recent year if active session year not found
        }
            
        $user = Auth::user();
        if ($user->isAdmin() || $user->isHeadTeacher()) {
            $this->classes = SchoolClass::orderBy('order')->get();
        } elseif ($user->isTeacher()) {
            $this->classes = $user->classes()->orderBy('order')->get();
        }
    }

    public function updatedClassId()
    {
        $this->batchId = '';
        $this->batches = [];
        $this->approvedResults = [];
        $this->isLoaded = false;
        
        if ($this->classId) {
            $this->batches = \App\Models\Batch::where('school_class_id', $this->classId)
                ->orderBy('name')
                ->get();
        }
    }

    public function updatedBatchId()
    {
        $this->approvedResults = [];
        $this->isLoaded = false;
    }

    public function loadResults()
    {
        if (!$this->selectedYear) {
            return;
        }

        $user = Auth::user();
        
        $query = AnnualResultApproval::with(['student.batch.schoolClass', 'approvedBy'])
            ->where('year', $this->selectedYear);

        // Filter by user permissions
        if ($user->isTeacher()) {
            $classIds = $user->classes()->pluck('school_classes.id');
            $query->whereHas('student.batch', function($q) use ($classIds) {
                $q->whereIn('school_class_id', $classIds);
            });
        }

        // Filter by class if selected
        if ($this->classId) {
            $query->whereHas('student.batch', function($q) {
                $q->where('school_class_id', $this->classId);
            });
        }

        // Filter by batch if selected
        if ($this->batchId) {
            $query->whereHas('student', function($q) {
                $q->where('batch_id', $this->batchId);
            });
        }

        // Filter by specific student if selected
        if ($this->filteredStudentId) {
            $query->where('student_id', $this->filteredStudentId);
        }

        $approvals = $query->orderBy('created_at', 'desc')->get();

        $this->approvedResults = $approvals->map(function($approval) {
            return [
                'id' => $approval->id,
                'student' => $approval->student,
                'class' => $approval->student->batch->schoolClass ?? null,
                'batch' => $approval->student->batch ?? null,
                'year' => $approval->year,
                'approved_at' => $approval->approved_at,
                'approved_by' => $approval->approvedBy,
            ];
        })->toArray();

        $this->isLoaded = true;
    }

    public function selectStudent($studentId)
    {
        $this->selectedStudent = User::with('batch.schoolClass')->find($studentId);
        $this->filteredStudentId = $studentId;
        
        if ($this->selectedStudent && $this->selectedStudent->batch) {
            $this->classId = $this->selectedStudent->batch->school_class_id;
            $this->updatedClassId();
            $this->batchId = $this->selectedStudent->batch_id;
            
            // Auto-load results if year is selected
            if ($this->selectedYear) {
                $this->loadResults();
            }
        }
    }
    
    public function clearFilter()
    {
        $this->studentSearch = '';
        $this->filteredStudentId = null;
        $this->selectedStudent = null;
    }
    
    public function getStudentsForSearch()
    {
        $user = Auth::user();
        
        if ($user->isAdmin() || $user->isHeadTeacher()) {
            return User::whereHas('batch')
                ->whereDoesntHave('roles')
                ->where(function($q) {
                    $q->where('firstname', 'like', '%' . $this->studentSearch . '%')
                      ->orWhere('middlename', 'like', '%' . $this->studentSearch . '%')
                      ->orWhere('surname', 'like', '%' . $this->studentSearch . '%')
                      ->orWhere('student_id', 'like', '%' . $this->studentSearch . '%');
                })
                ->with('batch.schoolClass')
                ->orderBy('firstname')
                ->limit(10)
                ->get();
        }
        
        if ($user->isTeacher()) {
            $classIds = $user->classes()->pluck('school_classes.id');
            
            return User::whereHas('batch', function($q) use ($classIds) {
                    $q->whereIn('school_class_id', $classIds);
                })
                ->whereDoesntHave('roles')
                ->where(function($q) {
                    $q->where('firstname', 'like', '%' . $this->studentSearch . '%')
                      ->orWhere('middlename', 'like', '%' . $this->studentSearch . '%')
                      ->orWhere('surname', 'like', '%' . $this->studentSearch . '%')
                      ->orWhere('student_id', 'like', '%' . $this->studentSearch . '%');
                })
                ->with('batch.schoolClass')
                ->orderBy('firstname')
                ->limit(10)
                ->get();
        }
        
        return collect();
    }

    public function unapproveResult($studentId, $year)
    {
        try {
            $student = User::find($studentId);
            if (!$student) {
                session()->flash('error', 'Student not found.');
                return;
            }

            $approval = \App\Models\AnnualResultApproval::where('student_id', $studentId)
                ->where('year', $year)
                ->first();

            if ($approval) {
                $approval->delete();
                session()->flash('success', "Annual result approval for {$student->name} has been removed.");
                $this->loadResults(); // Refresh the data
            } else {
                session()->flash('error', 'No approval found for this student.');
            }
        } catch (\Exception $e) {
            session()->flash('error', 'Error removing annual result approval.');
        }
    }
    
    public function canUnapprove()
    {
        $user = auth()->user();
        
        if ($user->isAdmin()) {
            return \App\Models\Setting::get('allow_admin_unapprove_results', false);
        }
        
        if ($user->isHeadTeacher()) {
            return \App\Models\Setting::get('allow_head_teacher_unapprove_results', false);
        }
        
        if ($user->isTeacher()) {
            return \App\Models\Setting::get('allow_teacher_unapprove_results', false);
        }
        
        return false;
    }

    public function viewStudentResult($studentId, $year)
    {
        $data = $this->annualResultService->getStudentAnnualResultForPrint($studentId, $year);
        
        if ($data) {
            $this->selectedStudentData = $data;
            $this->showingResultFor = $data['student']->name;
        }
    }
    
    public function closeStudentResult()
    {
        $this->selectedStudentData = null;
        $this->showingResultFor = null;
    }

    public function render()
    {
        return view('livewire.results.view-annual-results');
    }
}