<?php

namespace App\Livewire\Student;

use Livewire\Component;
use Livewire\WithPagination;
use App\Traits\HasNotifications;
use App\Models\PaymentRequest;
use App\Models\Payment;
use App\Models\Setting;

class StudentPayments extends Component
{
    use HasNotifications, WithPagination;

    public $showPaymentModal = false;
    public $showNotesModal = false;
    public $selectedPaymentRequest = null;
    public $selectedPayment = null;
    public $schoolAccounts = [];
    public $debugAccounts = '';
    public $hasPendingPayment = false;

    public function makePayment($paymentRequestId)
    {
        $this->selectedPaymentRequest = PaymentRequest::find($paymentRequestId);
        
        // Get school accounts from settings
        $accountsData = Setting::where('key', 'school_accounts')->first();
        $this->schoolAccounts = $accountsData ? json_decode($accountsData->value, true) : [];
        
        // Check if user has pending payment for this request
        $this->hasPendingPayment = Payment::where('payment_request_id', $paymentRequestId)
            ->where('user_id', auth()->id())
            ->where('status', 'pending')
            ->exists();
            
        $this->showPaymentModal = true;
    }
    
    public function closeModal()
    {
        $this->showPaymentModal = false;
        $this->selectedPaymentRequest = null;
        $this->schoolAccounts = [];
        $this->hasPendingPayment = false;
    }
    
    public function viewNotes($paymentId)
    {
        $this->selectedPayment = Payment::find($paymentId);
        $this->showNotesModal = true;
    }
    
    public function closeNotesModal()
    {
        $this->showNotesModal = false;
        $this->selectedPayment = null;
    }
    
    public function confirmPayment()
    {
        if ($this->hasPendingPayment) {
            $this->showToast('warning', 'You have already submitted a payment confirmation for this request.', 'Payment');
            return;
        }
        
        try {
            Payment::create([
                'payment_request_id' => $this->selectedPaymentRequest->id,
                'user_id' => auth()->id(),
                'status' => 'pending',
                'amount' => $this->selectedPaymentRequest->amount,
                'currency' => $this->selectedPaymentRequest->currency,
                'notes' => 'Student payment confirmation',
                'payment_date' => now(),
                'recorded_by' => auth()->id(),
            ]);
            
            $this->hasPendingPayment = true;
            $this->showToast('success', 'Payment confirmation submitted successfully! Please remember to make the transfer to one of the designated accounts.', 'Payment');
            
        } catch (\Exception $e) {
            $this->showToast('error', 'Error submitting payment confirmation: ' . $e->getMessage(), 'Payment');
        }
    }

    public function render()
    {
        $student = auth()->user();
        
        $paymentRequests = PaymentRequest::where('status', 'active')
        ->where(function($query) use ($student) {
            // Payments targeting this specific student
            $query->where(function($q) use ($student) {
                $q->where('target_type', 'student')
                  ->where('student_id', $student->id);
            })
            // Payments targeting student's class
            ->orWhere(function($q) use ($student) {
                $q->where('target_type', 'class')
                  ->where('school_class_id', $student->school_class_id);
            })
            // Payments targeting student's batch
            ->orWhere(function($q) use ($student) {
                $q->where('target_type', 'batch')
                  ->where('batch_id', $student->batch_id);
            })
            // Payments targeting all students
            ->orWhere('target_type', 'all_students');
        })
        ->whereDoesntHave('payments', function($query) use ($student) {
            $query->where('user_id', $student->id)
                  ->where('status', 'successful');
        })
        ->orderBy('created_at', 'desc')
        ->paginate(10);
        
        $paymentHistory = Payment::where('user_id', $student->id)
            ->with('paymentRequest')
            ->orderBy('created_at', 'desc')
            ->paginate(10, ['*'], 'historyPage');

        return view('livewire.student.student-payments', compact('paymentRequests', 'paymentHistory'));
    }
}