<?php

namespace App\Livewire;

use App\Models\Batch;
use App\Models\SchoolClass;
use App\Models\User;
use App\Traits\HasNotifications;
use Livewire\Component;
use Livewire\WithPagination;

class StudentBatchAssignment extends Component
{
    use HasNotifications, WithPagination;

    public $selectedClass = '';
    public $search = '';
    public $showUnassignedOnly = false;
    public $batches = [];
    public $selectedStudents = [];
    public $selectedBatch = '';
    public $perPage = 20;

    public function mount()
    {
        $this->loadBatches();
    }

    public function updatedSelectedClass()
    {
        $this->loadBatches();
        $this->resetPage();
    }

    public function updatedSearch()
    {
        $this->resetPage();
    }

    public function updatedShowUnassignedOnly()
    {
        $this->resetPage();
    }

    public function loadBatches()
    {
        if ($this->selectedClass) {
            $this->batches = Batch::where('school_class_id', $this->selectedClass)
                ->where('is_active', true)
                ->orderBy('name')
                ->get();
        } else {
            $this->batches = [];
        }
    }

    public function getStudentsProperty()
    {
        $query = User::whereDoesntHave('roles')
            ->with('batch.schoolClass')
            ->orderBy('firstname');

        if ($this->search) {
            $query->where(function ($q) {
                $q->where('firstname', 'like', '%' . $this->search . '%')
                  ->orWhere('surname', 'like', '%' . $this->search . '%')
                  ->orWhere('student_id', 'like', '%' . $this->search . '%');
            });
        }

        if ($this->showUnassignedOnly) {
            $query->whereNull('batch_id');
        } elseif ($this->selectedClass) {
            $query->where(function ($q) {
                $q->whereHas('batch', function ($subQ) {
                    $subQ->where('school_class_id', $this->selectedClass);
                })->orWhereNull('batch_id');
            });
        }

        return $query->paginate($this->perPage);
    }

    public function assignStudent($studentId, $batchId)
    {
        try {
            $student = User::findOrFail($studentId);
            $batch = Batch::findOrFail($batchId);
            
            if ($batch->current_students >= $batch->capacity) {
                $this->showError('Batch is at full capacity.');
                return;
            }

            $student->update(['batch_id' => $batchId]);
            $this->showSuccess("Student {$student->name} assigned to {$batch->name} successfully!");
        } catch (\Exception $e) {
            $this->showError('An error occurred while assigning student.');
        }
    }

    public function removeStudent($studentId)
    {
        try {
            $student = User::findOrFail($studentId);
            $student->update(['batch_id' => null]);
            $this->showSuccess("Student {$student->name} removed from batch successfully!");
        } catch (\Exception $e) {
            $this->showError('An error occurred while removing student.');
        }
    }

    public function selectAll()
    {
        $this->selectedStudents = $this->students->pluck('id')->toArray();
    }

    public function deselectAll()
    {
        $this->selectedStudents = [];
    }

    public function updatedSelectedStudents()
    {
        // This method ensures the component re-renders when selectedStudents changes
        // No additional logic needed, just triggers reactivity
    }

    public function massAssign()
    {
        if (empty($this->selectedStudents) || !$this->selectedBatch) {
            $this->showError('Please select students and a batch.');
            return;
        }

        try {
            $batch = Batch::findOrFail($this->selectedBatch);
            $assignedCount = 0;
            $errors = [];

            foreach ($this->selectedStudents as $studentId) {
                $student = User::find($studentId);
                if ($student && $batch->current_students < $batch->capacity) {
                    $student->update(['batch_id' => $this->selectedBatch]);
                    $assignedCount++;
                } else {
                    $errors[] = $student ? $student->name : 'Unknown student';
                }
            }

            if ($assignedCount > 0) {
                $this->showSuccess("Successfully assigned {$assignedCount} students to {$batch->name}.");
            }
            
            if (!empty($errors)) {
                $this->showError('Some students could not be assigned due to batch capacity.');
            }

            $this->selectedStudents = [];
            $this->selectedBatch = '';
        } catch (\Exception $e) {
            $this->showError('An error occurred during mass assignment.');
        }
    }

    public function massDetach()
    {
        if (empty($this->selectedStudents)) {
            $this->showError('Please select students to detach.');
            return;
        }

        try {
            $detachedCount = 0;
            foreach ($this->selectedStudents as $studentId) {
                $student = User::find($studentId);
                if ($student && $student->batch_id) {
                    $student->update(['batch_id' => null]);
                    $detachedCount++;
                }
            }

            if ($detachedCount > 0) {
                $this->showSuccess("Successfully detached {$detachedCount} students from their batches.");
            } else {
                $this->showError('No students were detached (they may already be unassigned).');
            }

            $this->selectedStudents = [];
        } catch (\Exception $e) {
            $this->showError('An error occurred during mass detachment.');
        }
    }

    public function render()
    {
        $schoolClasses = SchoolClass::where('is_active', true)->orderBy('order')->get();
        $allBatches = Batch::where('is_active', true)->with('schoolClass')->orderBy('name')->get();
        $students = $this->students;
        return view('livewire.student-batch-assignment', compact('schoolClasses', 'allBatches', 'students'));
    }
}
