<?php

namespace App\Livewire;

use App\Models\User;
use App\Models\Batch;
use App\Models\Country;
use App\Models\State;
use App\Models\City;
use App\Services\StudentService;
use App\Traits\HasNotifications;
use Illuminate\Support\Facades\Hash;
use Livewire\Component;
use Livewire\WithFileUploads;
use Illuminate\Support\Facades\Storage;

class StudentForm extends Component
{
    use HasNotifications, WithFileUploads;

    public $studentId = null;
    public $currentTab = 'basic';
    public $firstname = '';
    public $middlename = '';
    public $surname = '';
    public $email = '';
    public $phone = '';
    public $student_id = '';
    public $gender = '';
    public $address = '';
    public $dob = '';
    public $place_of_birth = '';
    public $previous_school = '';
    public $previous_class = '';
    public $admission_date = '';
    public $nok_name = '';
    public $nok_phone = '';
    public $nok_relationship = '';
    public $nok_address = '';
    public $official_remarks = '';
    public $school_class_id = '';
    public $batch_id = '';
    public $active = true;
    public $new_password = '';
    public $confirm_password = '';
    public $enable_email_edit = false;
    public $enable_student_id_edit = false;
    public $country_id = '';
    public $state_id = '';
    public $city_id = '';
    public $countries = [];
    public $states = [];
    public $cities = [];
    public $countriesLoaded = false;
    public $statesLoaded = false;
    public $citiesLoaded = false;
    public $loadingCountries = false;
    public $loadingStates = false;
    public $loadingCities = false;
    public $passport_photo;
    public $existing_profile_photo = '';

    protected $rules = [
        'firstname' => 'required|string|max:255',
        'middlename' => 'nullable|string|max:255',
        'surname' => 'required|string|max:255',
        'email' => 'nullable|email|unique:users,email',
        'phone' => 'nullable|string|max:20',
        'student_id' => 'nullable|string|size:10|unique:users,student_id',
        'gender' => 'required|in:male,female',
        'address' => 'nullable|string',
        'dob' => 'required|date',
        'place_of_birth' => 'nullable|string|max:255',
        'previous_school' => 'nullable|string|max:255',
        'previous_class' => 'nullable|string|max:255',
        'admission_date' => 'required|date',
        'nok_name' => 'nullable|string|max:255',
        'nok_phone' => 'required|string|max:20',
        'nok_relationship' => 'nullable|string|max:255',
        'nok_address' => 'nullable|string',
        'official_remarks' => 'nullable|string',
        'school_class_id' => 'required|exists:school_classes,id',
        'batch_id' => 'required|exists:batches,id',
        'country_id' => 'nullable|integer',
        'state_id' => 'nullable|integer',
        'city_id' => 'nullable|integer',
        'passport_photo' => 'nullable|image|max:500',
    ];

    public function mount($studentId = null)
    {
        if ($studentId) {
            $student = User::where('student_id', $studentId)->firstOrFail();
            $this->studentId = $student->id;
            $this->fill($student->toArray());
            $this->active = $student->active ?? true;
            $this->existing_profile_photo = $student->profile_photo;
            
            if ($this->country_id) {
                $this->loadCountries();
                $this->loadStates();
            }
            if ($this->state_id) {
                $this->loadCities();
            }
        } else {
            $this->admission_date = date('Y-m-d');
        }
    }

    public function updated($propertyName)
    {
        // Skip validation for email and student_id if editing is disabled
        if ($this->studentId && $propertyName === 'email' && !$this->enable_email_edit) {
            return;
        }
        
        if ($this->studentId && $propertyName === 'student_id' && !$this->enable_student_id_edit) {
            return;
        }
        
        // Create dynamic rules for validation
        $rules = $this->rules;
        if ($this->studentId) {
            if ($propertyName === 'email') {
                $rules['email'] = 'nullable|email|unique:users,email,' . $this->studentId;
            }
            if ($propertyName === 'student_id') {
                $rules['student_id'] = 'nullable|string|size:10|unique:users,student_id,' . $this->studentId;
            }
        }
        
        $this->validateOnly($propertyName, $rules);
        
        if ($propertyName === 'country_id') {
            $this->loadStates();
        }
        
        if ($propertyName === 'state_id') {
            $this->loadCities();
        }
        
        if ($propertyName === 'school_class_id') {
            $this->batch_id = '';
        }
    }
    
    public function loadCountries()
    {
        if (!$this->countriesLoaded) {
            $this->loadingCountries = true;
            $this->countries = Country::orderBy('name')->get();
            $this->countriesLoaded = true;
            $this->loadingCountries = false;
        }
    }
    
    public function loadStates()
    {
        if ($this->country_id) {
            $this->loadingStates = true;
            $this->state_id = '';
            $this->city_id = '';
            $this->states = State::where('country_id', $this->country_id)->orderBy('name')->get();
            $this->cities = [];
            $this->statesLoaded = true;
            $this->citiesLoaded = false;
            $this->loadingStates = false;
        }
    }
    
    public function loadCities()
    {
        if ($this->state_id) {
            $this->loadingCities = true;
            $this->city_id = '';
            $this->cities = City::where('state_id', $this->state_id)->orderBy('name')->get();
            $this->citiesLoaded = true;
            $this->loadingCities = false;
        }
    }

    public function switchTab($tab)
    {
        $this->currentTab = $tab;
    }

    public function saveBasic()
    {
        $basicRules = [
            'firstname' => 'required|string|max:255',
            'middlename' => 'nullable|string|max:255',
            'surname' => 'required|string|max:255',
            'gender' => 'required|in:male,female',
            'admission_date' => 'required|date',
            'nok_phone' => 'required|string|max:20',
            'dob' => 'required|date',
        ];

        if ($this->studentId) {
            // For existing students, handle email and student_id conditionally
            if ($this->enable_email_edit) {
                $basicRules['email'] = 'nullable|email|unique:users,email,' . $this->studentId;
            }
            
            if ($this->enable_student_id_edit) {
                $basicRules['student_id'] = 'nullable|string|size:10|unique:users,student_id,' . $this->studentId;
            }
        } else {
            // For new students, require class, batch, email and student_id
            $basicRules['email'] = 'nullable|email|unique:users,email';
            $basicRules['student_id'] = 'nullable|string|size:10|unique:users,student_id';
            $basicRules['school_class_id'] = 'required|exists:school_classes,id';
            $basicRules['batch_id'] = 'required|exists:batches,id';
        }

        $this->validate($basicRules);

        try {
            $data = [
                'firstname' => $this->firstname,
                'middlename' => $this->middlename,
                'surname' => $this->surname,
                'gender' => $this->gender,
                'admission_date' => $this->admission_date,
                'nok_phone' => $this->nok_phone,
                'dob' => $this->dob,
            ];
            
            // Only include batch for new students (class comes through batch relationship)
            if (!$this->studentId) {
                $data['batch_id'] = $this->batch_id;
                $data['email'] = $this->email;
                $data['student_id'] = $this->student_id;
            } else {
                // For existing students, only include email/student_id if editing is enabled
                if ($this->enable_email_edit) {
                    $data['email'] = $this->email;
                }
                
                if ($this->enable_student_id_edit) {
                    $data['student_id'] = $this->student_id;
                }
            }

            $studentService = app(StudentService::class);

            if ($this->studentId) {
                $student = User::findOrFail($this->studentId);
                $studentService->updateStudent($student, $data);
                $message = 'Basic information updated successfully';
            } else {
                $student = $studentService->createStudent($data);
                $this->studentId = $student->id;
                $this->email = $student->email;
                $this->student_id = $student->student_id;
                $message = 'Student created successfully';
            }

            $this->showToast('success', $message, 'Student Information');
            $this->currentTab = 'personal';

        } catch (\Exception $e) {
            $this->showToast('error', 'Error: ' . $e->getMessage(), 'Student Information');
        }
    }

    public function savePersonal()
    {
        if (!$this->studentId) {
            $this->showToast('error', 'Please complete basic information first', 'Student Information');
            return;
        }

        $personalRules = [
            'phone' => 'nullable|string|max:20',
            'address' => 'nullable|string',
            'place_of_birth' => 'nullable|string|max:255',
            'country_id' => 'nullable|integer',
            'state_id' => 'nullable|integer',
            'city_id' => 'nullable|integer',
            'passport_photo' => 'nullable|image|max:500',
        ];

        $this->validate($personalRules);

        try {
            $student = User::findOrFail($this->studentId);
            
            $updateData = [
                'phone' => $this->phone,
                'address' => $this->address,
                'place_of_birth' => $this->place_of_birth,
                'country_id' => $this->country_id ?: null,
                'state_id' => $this->state_id ?: null,
                'city_id' => $this->city_id ?: null,
            ];
            
            if ($this->passport_photo) {
                // Delete old profile photo if exists
                if ($this->existing_profile_photo && Storage::disk('public')->exists($this->existing_profile_photo)) {
                    Storage::disk('public')->delete($this->existing_profile_photo);
                }
                
                $filename = 'passport_' . $student->student_id . '.' . $this->passport_photo->getClientOriginalExtension();
                $path = $this->passport_photo->storeAs('passport_photos', $filename, 'public');
                $updateData['profile_photo'] = $path;
                $this->existing_profile_photo = $path;
                $this->passport_photo = null; // Clear the uploaded file
            }
            
            $student->update($updateData);

            $this->showToast('success', 'Personal information updated successfully', 'Student Information');
            $this->currentTab = 'academic';

        } catch (\Exception $e) {
            $this->showToast('error', 'Error: ' . $e->getMessage(), 'Student Information');
        }
    }

    public function saveAcademic()
    {
        if (!$this->studentId) {
            $this->showToast('error', 'Please complete basic information first', 'Student Information');
            return;
        }

        $academicRules = [
            'previous_school' => 'nullable|string|max:255',
            'previous_class' => 'nullable|string|max:255',
        ];

        $this->validate($academicRules);

        try {
            $student = User::findOrFail($this->studentId);
            $student->update([
                'previous_school' => $this->previous_school,
                'previous_class' => $this->previous_class,
            ]);

            $this->showToast('success', 'Academic information updated successfully', 'Student Information');
            $this->currentTab = 'guardian';

        } catch (\Exception $e) {
            $this->showToast('error', 'Error: ' . $e->getMessage(), 'Student Information');
        }
    }

    public function saveGuardian()
    {
        if (!$this->studentId) {
            $this->showToast('error', 'Please complete basic information first', 'Student Information');
            return;
        }

        $guardianRules = [
            'nok_name' => 'nullable|string|max:255',
            'nok_relationship' => 'nullable|string|max:255',
            'nok_address' => 'nullable|string',
            'official_remarks' => 'nullable|string',
        ];

        $this->validate($guardianRules);

        try {
            $student = User::findOrFail($this->studentId);
            $student->update([
                'nok_name' => $this->nok_name,
                'nok_relationship' => $this->nok_relationship,
                'nok_address' => $this->nok_address,
                'official_remarks' => $this->official_remarks,
            ]);

            session()->flash('toast', [
                'type' => 'success',
                'message' => 'Student registration completed successfully'
            ]);

            return redirect()->route('students.index');

        } catch (\Exception $e) {
            session()->flash('toast', [
                'type' => 'error',
                'message' => 'Error: ' . $e->getMessage()
            ]);
        }
    }

    public function deleteStudent()
    {
        if (!$this->studentId) return;

        try {
            $student = User::findOrFail($this->studentId);
            $studentName = $student->name;
            
            app(StudentService::class)->deleteStudent($student);
            
            session()->flash('toast', [
                'type' => 'success',
                'message' => "Student {$studentName} deleted successfully"
            ]);
            
            return redirect()->route('students.index');
        } catch (\Exception $e) {
            session()->flash('toast', [
                'type' => 'error',
                'message' => 'Failed to delete student: ' . $e->getMessage()
            ]);
        }
    }

    public function toggleStudentStatus()
    {
        if (!$this->studentId) return;

        try {
            $student = User::findOrFail($this->studentId);
            $newStatus = !$student->active;
            $student->update(['active' => $newStatus]);
            $this->active = $newStatus;
            
            $status = $newStatus ? 'activated' : 'blocked';
            $this->showToast('success', "Student access {$status} successfully", 'Student Security');
        } catch (\Exception $e) {
            $this->showToast('error', 'Failed to update student status: ' . $e->getMessage(), 'Student Security');
        }
    }

    public function resetPassword()
    {
        if (!$this->studentId) return;

        try {
            $student = User::findOrFail($this->studentId);
            $defaultPassword = substr($student->student_id, -4);
            $student->update(['password' => Hash::make($defaultPassword)]);
            
            $this->showToast('success', 'Password reset to default successfully', 'Student Security');
        } catch (\Exception $e) {
            $this->showToast('error', 'Failed to reset password: ' . $e->getMessage(), 'Student Security');
        }
    }

    public function setCustomPassword()
    {
        if (!$this->studentId) return;

        $this->validate([
            'new_password' => 'required|min:6',
            'confirm_password' => 'required|same:new_password',
        ]);

        try {
            $student = User::findOrFail($this->studentId);
            $student->update(['password' => Hash::make($this->new_password)]);
            
            $this->new_password = '';
            $this->confirm_password = '';
            
            $this->showToast('success', 'Custom password set successfully', 'Student Security');
        } catch (\Exception $e) {
            $this->showToast('error', 'Failed to set password: ' . $e->getMessage(), 'Student Security');
        }
    }

    public function render()
    {
        $schoolClasses = \App\Models\SchoolClass::where('is_active', true)->orderBy('order')->get();
        $batches = $this->school_class_id 
            ? Batch::where('school_class_id', $this->school_class_id)->where('is_active', true)->orderBy('name')->get()
            : collect();
        return view('livewire.student-form', compact('schoolClasses', 'batches'));
    }
}