<?php

namespace App\Livewire;

use App\Models\User;
use App\Models\SchoolClass;
use App\Traits\HasNotifications;
use Livewire\Component;
use Illuminate\Support\Facades\Cache;

class TeacherClassAssignment extends Component
{
    use HasNotifications;

    public $selectedTeacher = '';
    public $selectedClasses = [];
    
    protected $listeners = ['action-confirmed' => 'handleConfirmedAction'];
    public $pendingAssignment = false;

    public function mount()
    {
        $this->selectedClasses = [];
    }

    public function updatedSelectedTeacher()
    {
        if ($this->selectedTeacher) {
            $teacher = User::find($this->selectedTeacher);
            $this->selectedClasses = $teacher->classes->pluck('id')->toArray();
        } else {
            $this->selectedClasses = [];
        }
    }

    public function assignClasses()
    {
        $this->validate([
            'selectedTeacher' => 'required|exists:users,id',
            'selectedClasses' => 'array',
        ]);

        $teacher = User::find($this->selectedTeacher);
        $selectedClasses = SchoolClass::whereIn('id', $this->selectedClasses)
            ->orderBy('order')
            ->get();
        $selectedClassNames = $selectedClasses->map(function($class) {
            return $class->display_name;
        })->toArray();
        
        $classCount = count($selectedClassNames);
        $classList = implode(', ', $selectedClassNames);
        
        $message = "Are you sure you want to assign <strong>{$teacher->name}</strong> to ";
        $message .= $classCount === 1 ? "this class?" : "these {$classCount} classes?";
        $message .= "<br><br><strong>Classes:</strong> {$classList}";
        
        $this->pendingAssignment = true;
        $this->confirmAction('Assign Teacher to Classes', $message);
    }
    
    public function handleConfirmedAction()
    {
        if ($this->pendingAssignment) {
            $this->performAssignment();
            $this->pendingAssignment = false;
        }
    }
    
    private function performAssignment()
    {
        try {
            $teacher = User::find($this->selectedTeacher);
            $teacher->classes()->sync($this->selectedClasses);

            // Clear teacher-specific cache
            Cache::forget("teacher_students_count_{$teacher->id}");
            Cache::forget("teacher_classes_count_{$teacher->id}");

            $this->showToast('success', 'Classes assigned successfully', 'Teacher Assignment');
            
        } catch (\Exception $e) {
            $this->showToast('error', 'Error: ' . $e->getMessage(), 'Teacher Assignment');
        }
    }

    public function render()
    {
        $teachers = User::whereHas('roles', function($query) {
            $query->whereIn('name', ['teacher', 'head_teacher']);
        })->orderBy('firstname')->get();
        
        $classes = SchoolClass::orderBy('order')->get();

        return view('livewire.teacher-class-assignment', compact('teachers', 'classes'));
    }
}