<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use App\Helpers\CurrencyHelper;

class PaymentRequest extends Model
{
    use SoftDeletes;

    protected $fillable = [
        'payment_id',
        'title',
        'description',
        'amount',
        'currency',
        'status',
        'amount_paid',
        'balance',
        'meta',
        'target_type',
        'target_id',
        'student_id',
        'school_class_id',
        'batch_id',
        'created_by',
    ];
    
    protected static function boot()
    {
        parent::boot();
        
        static::creating(function ($paymentRequest) {
            if (empty($paymentRequest->payment_id)) {
                $paymentRequest->payment_id = static::generateUniquePaymentId();
            }
            if (empty($paymentRequest->status)) {
                $paymentRequest->status = 'active';
            }
        });
    }
    
    private static function generateUniquePaymentId()
    {
        do {
            $id = strtoupper(substr(str_shuffle('0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZ'), 0, 10));
        } while (static::where('payment_id', $id)->exists());
        
        return $id;
    }

    protected $casts = [
        'meta' => 'array',
        'amount' => 'decimal:2',
        'amount_paid' => 'decimal:2',
        'balance' => 'decimal:2',
    ];

    public function student()
    {
        return $this->belongsTo(User::class, 'student_id');
    }

    public function schoolClass()
    {
        return $this->belongsTo(SchoolClass::class, 'school_class_id');
    }

    public function batch()
    {
        return $this->belongsTo(Batch::class, 'batch_id');
    }

    public function createdBy()
    {
        return $this->belongsTo(User::class, 'created_by');
    }

    public function payments()
    {
        return $this->hasMany(Payment::class);
    }

    public function getCurrencySymbolAttribute()
    {
        return CurrencyHelper::getSymbol($this->currency);
    }
    
    public function getTargetNameAttribute()
    {
        switch ($this->target_type) {
            case 'student':
                return $this->student?->name;
            case 'class':
                return $this->schoolClass?->name;
            case 'batch':
                return $this->batch?->name;
            case 'all_students':
                return 'All Students';
            default:
                return 'Unknown';
        }
    }
}
