<?php

namespace App\Services;

use App\Models\Result;
use App\Models\User;
use App\Models\SchoolClass;
use App\Models\SchoolSession;
use App\Models\Subject;
use App\Models\SubjectAlias;
use App\Models\Setting;
use Illuminate\Support\Collection;

class AnnualResultService
{
    public function getAnnualResultData($year, $classId, $batchId = null)
    {
        $sessions = $this->getSessionsForYear($year);
        $class = SchoolClass::find($classId);
        $batch = $batchId ? \App\Models\Batch::find($batchId) : null;

        if ($sessions->isEmpty() || !$class || ($batchId && !$batch)) {
            return null;
        }

        $students = $batchId ? $this->getStudentsByBatch($batchId) : $this->getStudentsByClass($classId);

        // Try to get subjects from approval snapshots first
        $subjects = $this->getSubjectsFromApprovals($students, $sessions->pluck('id'));
        if (!$subjects) {
            $subjects = $this->getSubjectsByClass($classId);
        }

        $results = $batchId ? $this->getAnnualResultsForBatch($sessions->pluck('id'), $batchId) : $this->getAnnualResultsForClass($sessions->pluck('id'), $classId);

        $annualData = $this->calculateAnnualData($students, $subjects, $results, $sessions);

        return [
            'year' => $year,
            'sessions' => $sessions,
            'class' => $class,
            'batch' => $batch,
            'subjects' => $subjects,
            'annualData' => $annualData,
            'teachers' => $class->teachers,
            'generated_at' => now(),
            'max_total_average' => 100 * $subjects->count(),
            'school_info' => $this->getSchoolInfo()
        ];
    }

    public function getAnnualMasterListData($year, $classId, $batchId = null)
    {
        $sessions = $this->getSessionsForYear($year);
        $class = SchoolClass::with('teachers')->find($classId);
        $batch = $batchId ? \App\Models\Batch::find($batchId) : null;

        if ($sessions->isEmpty() || !$class || ($batchId && !$batch)) {
            return null;
        }

        $students = $batchId ? $this->getStudentsByBatch($batchId) : $this->getStudentsByClass($classId);

        // Try to get subjects from approval snapshots first
        $subjects = $this->getSubjectsFromApprovals($students, $sessions->pluck('id'));
        if (!$subjects) {
            $subjects = $this->getSubjectsByClass($classId);
        }

        $results = $batchId ? $this->getAnnualResultsForBatch($sessions->pluck('id'), $batchId) : $this->getAnnualResultsForClass($sessions->pluck('id'), $classId);

        $masterListData = $this->calculateAnnualMasterListData($students, $subjects, $results, $sessions);

        return [
            'year' => $year,
            'sessions' => $sessions,
            'class' => $class,
            'batch' => $batch,
            'subjects' => $subjects,
            'masterListData' => $masterListData,
            'teachers' => $class->teachers,
            'generated_at' => now(),
            'total_possible_score' => $subjects->count() * 100 * $sessions->count(),
            'max_annual_average' => 100,
            'max_total_average' => 100 * $subjects->count(),
            'school_info' => $this->getSchoolInfo()
        ];
    }

    private function getSessionsForYear($year)
    {
        return SchoolSession::where('start_year', $year)
            ->where('end_year', $year + 1)
            ->orderBy('start_year', 'DESC')
            ->get();
    }

    private function getStudentsByClass($classId)
    {
        return User::whereHas('batch', function ($query) use ($classId) {
            $query->where('school_class_id', $classId)
                  ->where('graduated', false);
        })->where('active', true)->orderBy('firstname')->get();
    }

    private function getStudentsByBatch($batchId)
    {
        return User::where('batch_id', $batchId)
            ->where('active', true)
            ->orderBy('firstname')
            ->get();
    }

    private function getSubjectsByClass($classId)
    {
        $class = SchoolClass::with(['subjects', 'subjectAliases'])->find($classId);

        return $class->subjects->map(function ($subject) use ($class) {
            $alias = $class->subjectAliases->where('subject_id', $subject->id)->first();
            $subject->display_name = $alias ? $alias->alias : $subject->name;
            return $subject;
        });
    }

    private function getSubjectsFromApprovals($students, $sessionIds)
    {
        $allSubjects = collect();

        // Collect subjects from all session snapshots
        foreach ($students as $student) {
            foreach ($sessionIds as $sessionId) {
                $approval = \App\Models\ResultApproval::where('student_id', $student->id)
                    ->where('session_id', $sessionId)
                    ->whereNotNull('subjects_snapshot')
                    ->first();

                if ($approval && $approval->subjects_snapshot) {
                    foreach ($approval->subjects_snapshot as $subjectData) {
                        // Add subject if not already in collection
                        if (!$allSubjects->contains('id', $subjectData['id'])) {
                            $allSubjects->push((object)[
                                'id' => $subjectData['id'],
                                'name' => $subjectData['name'],
                                'display_name' => $subjectData['display_name'],
                            ]);
                        }
                    }
                }
            }
        }

        return $allSubjects->isNotEmpty() ? $allSubjects->sortBy('id') : null;
    }

    private function getAnnualResultsForClass($sessionIds, $classId)
    {
        $studentIds = $this->getStudentsByClass($classId)->pluck('id');

        return Result::with(['student', 'subject', 'session'])
            ->whereIn('session_id', $sessionIds)
            ->whereIn('student_id', $studentIds)
            ->where('approved', true)
            ->get()
            ->groupBy(['student_id', 'session_id']);
    }

    private function getAnnualResultsForBatch($sessionIds, $batchId)
    {
        $studentIds = $this->getStudentsByBatch($batchId)->pluck('id');

        return Result::with(['student', 'subject', 'session'])
            ->whereIn('session_id', $sessionIds)
            ->whereIn('student_id', $studentIds)
            ->where('approved', true)
            ->get()
            ->groupBy(['student_id', 'session_id']);
    }

    private function calculateAnnualData($students, $subjects, $results, $sessions)
    {
        $annualData = [];

        foreach ($students as $student) {
            $studentResults = $results->get($student->id, collect());
            $studentData = [
                'student' => $student,
                'sessions' => [],
                'annual_grand_total' => 0,
                'annual_average' => 0,
                'annual_grades' => ['A' => 0, 'B' => 0, 'C' => 0, 'D' => 0, 'E' => 0, 'F' => 0],
                'session_count' => 0
            ];

            foreach ($sessions as $session) {
                $sessionResults = $studentResults->get($session->id, collect());
                $sessionData = [
                    'session' => $session,
                    'subjects' => [],
                    'grand_total' => 0,
                    'average' => 0,
                    'total_possible_score' => 0
                ];

                $subjectsOfferedInSession = 0;

                foreach ($subjects as $subject) {
                    $result = $sessionResults->where('subject_id', $subject->id)->first();

                    // Check if subject was offered in this session by looking at approval snapshot
                    $approval = \App\Models\ResultApproval::where('student_id', $student->id)
                        ->where('session_id', $session->id)
                        ->whereNotNull('subjects_snapshot')
                        ->first();

                    $subjectOffered = true; // Default to true if no approval snapshot found
                    if ($approval && $approval->subjects_snapshot) {
                        $subjectOffered = false; // Reset to false, then check if subject exists in snapshot
                        foreach ($approval->subjects_snapshot as $subjectData) {
                            if ($subjectData['id'] == $subject->id) {
                                $subjectOffered = true;
                                break;
                            }
                        }
                    }

                    if (!$subjectOffered) {
                        // Subject not offered in this session
                        $sessionData['subjects'][$subject->id] = [
                            'ca_score' => '-',
                            'exam_score' => '-',
                            'total' => '-',
                            'grade' => '-'
                        ];
                    } elseif ($result) {
                        // Subject offered and student has result
                        $total = $result->total_score;
                        $grade = $this->calculateGrade($total, $session);

                        $sessionData['subjects'][$subject->id] = [
                            'ca_score' => $result->ca_score,
                            'exam_score' => $result->exam_score,
                            'total' => $total,
                            'grade' => $grade
                        ];
                        $sessionData['grand_total'] += $total;
                        $studentData['annual_grades'][$grade]++;
                        $subjectsOfferedInSession++;
                    } else {
                        // Subject offered but no result (student failed/absent)
                        $sessionData['subjects'][$subject->id] = [
                            'ca_score' => 0,
                            'exam_score' => 0,
                            'total' => 0,
                            'grade' => 'F'
                        ];
                        $studentData['annual_grades']['F']++;
                        $subjectsOfferedInSession++;
                    }
                }

                $sessionData['total_possible_score'] = $subjectsOfferedInSession * 100;
                $sessionData['average'] = $subjectsOfferedInSession > 0 ?
                    round(($sessionData['grand_total'] / ($subjectsOfferedInSession * 100)) * 100, 2) : 0;

                $studentData['annual_grand_total'] += $sessionData['grand_total'];
                $studentData['session_count']++;

                $studentData['sessions'][$session->id] = $sessionData;
            }

            // Calculate total possible score and annual average as percentage
            $totalPossibleScore = $subjects->count() * 100 * $sessions->count();
            $studentData['total_possible_score'] = $totalPossibleScore;
            $studentData['annual_average'] = $totalPossibleScore > 0 ?
                round(($studentData['annual_grand_total'] / $totalPossibleScore) * 100, 2) : 0;

            // Calculate total average: sum of subject averages across all sessions
            $subjectTotalAverages = 0;
            foreach ($subjects as $subject) {
                $subjectTotal = 0;
                foreach ($studentData['sessions'] as $sessionData) {
                    $total = $sessionData['subjects'][$subject->id]['total'] ?? 0;
                    $subjectTotal += ($total === '-' || $total === 'N/A') ? 0 : $total;
                }
                $subjectAverage = $sessions->count() > 0 ? $subjectTotal / $sessions->count() : 0;
                $subjectTotalAverages += $subjectAverage;
            }
            $studentData['total_average'] = round($subjectTotalAverages, 2);

            $annualData[] = $studentData;
        }

        // Sort by annual grand total (descending) and calculate positions
        usort($annualData, function ($a, $b) {
            return $b['annual_grand_total'] <=> $a['annual_grand_total'];
        });

        // Add position and position suffix with tie handling
        $currentPosition = 1;
        $previousTotal = null;
        foreach ($annualData as $index => &$studentData) {
            if ($previousTotal !== null && $studentData['annual_grand_total'] < $previousTotal) {
                $currentPosition = $index + 1;
            }
            $studentData['position'] = $currentPosition;
            $studentData['position_suffix'] = $this->getPositionSuffix($currentPosition);
            $previousTotal = $studentData['annual_grand_total'];
        }

        return $annualData;
    }

    private function calculateAnnualMasterListData($students, $subjects, $results, $sessions)
    {
        $masterList = [];

        foreach ($students as $student) {
            $studentResults = $results->get($student->id, collect());
            $studentData = [
                'student' => $student,
                'sessions' => [],
                'annual_grand_total' => 0,
                'annual_average' => 0,
                'annual_grades' => ['A' => 0, 'B' => 0, 'C' => 0, 'D' => 0, 'E' => 0, 'F' => 0],
                'position' => 0,
                'remark' => 'Fail',
                'session_count' => 0
            ];

            foreach ($sessions as $session) {
                $sessionResults = $studentResults->get($session->id, collect());
                $sessionData = [
                    'session' => $session,
                    'subjects' => [],
                    'grand_total' => 0,
                    'average' => 0,
                    'grades' => ['A' => 0, 'B' => 0, 'C' => 0, 'D' => 0, 'E' => 0, 'F' => 0]
                ];

                $subjectsOfferedInSession = 0;

                foreach ($subjects as $subject) {
                    $result = $sessionResults->where('subject_id', $subject->id)->first();

                    // Check if subject was offered in this session
                    $approval = \App\Models\ResultApproval::where('student_id', $student->id)
                        ->where('session_id', $session->id)
                        ->whereNotNull('subjects_snapshot')
                        ->first();

                    $subjectOffered = false;
                    if ($approval && $approval->subjects_snapshot) {
                        foreach ($approval->subjects_snapshot as $subjectData) {
                            if ($subjectData['id'] == $subject->id) {
                                $subjectOffered = true;
                                break;
                            }
                        }
                    }

                    if (!$subjectOffered) {
                        // Subject not offered in this session
                        $sessionData['subjects'][$subject->id] = [
                            'ca_score' => '-',
                            'exam_score' => '-',
                            'total' => '-',
                            'grade' => '-'
                        ];
                    } elseif ($result) {
                        // Subject offered and student has result
                        $total = $result->total_score;
                        $grade = $this->calculateGrade($total, $session);

                        $sessionData['subjects'][$subject->id] = [
                            'ca_score' => $result->ca_score,
                            'exam_score' => $result->exam_score,
                            'total' => $total,
                            'grade' => $grade
                        ];

                        $sessionData['grand_total'] += $total;
                        $sessionData['grades'][$grade]++;
                        $studentData['annual_grades'][$grade]++;
                        $subjectsOfferedInSession++;
                    } else {
                        // Subject offered but no result (student failed/absent)
                        $sessionData['subjects'][$subject->id] = [
                            'ca_score' => 0,
                            'exam_score' => 0,
                            'total' => 0,
                            'grade' => 'F'
                        ];
                        $sessionData['grades']['F']++;
                        $studentData['annual_grades']['F']++;
                        $subjectsOfferedInSession++;
                    }
                }

                $sessionData['average'] = $subjectsOfferedInSession > 0 ?
                    round($sessionData['grand_total'] / $subjectsOfferedInSession, 2) : 0;

                if ($sessionData['grand_total'] > 0) {
                    $studentData['annual_grand_total'] += $sessionData['grand_total'];
                    $studentData['session_count']++;
                }

                $studentData['sessions'][$session->id] = $sessionData;
            }

            // Calculate total possible score and annual average as percentage
            $totalPossibleScore = $subjects->count() * 100 * $sessions->count();
            $studentData['total_possible_score'] = $totalPossibleScore;
            $studentData['annual_average'] = $totalPossibleScore > 0 ?
                round(($studentData['annual_grand_total'] / $totalPossibleScore) * 100, 2) : 0;

            // Calculate total average: sum of subject averages across all sessions
            $subjectTotalAverages = 0;
            foreach ($subjects as $subject) {
                $subjectTotal = 0;
                foreach ($studentData['sessions'] as $sessionData) {
                    $total = $sessionData['subjects'][$subject->id]['total'] ?? 0;
                    $subjectTotal += ($total === '-' || $total === 'N/A') ? 0 : $total;
                }
                $subjectAverage = $sessions->count() > 0 ? $subjectTotal / $sessions->count() : 0;
                $subjectTotalAverages += $subjectAverage;
            }
            $studentData['total_average'] = round($subjectTotalAverages, 2);

            $masterList[] = $studentData;
        }

        // Sort by annual grand total (descending) for position calculation
        usort($masterList, function ($a, $b) {
            return $b['annual_grand_total'] <=> $a['annual_grand_total'];
        });

        // Calculate positions and remarks with tie handling
        $currentPosition = 1;
        $previousTotal = null;
        foreach ($masterList as $index => &$studentData) {
            if ($previousTotal !== null && $studentData['annual_grand_total'] < $previousTotal) {
                $currentPosition = $index + 1;
            }
            $studentData['position'] = $currentPosition;
            $studentData['position_suffix'] = $this->getPositionSuffix($currentPosition);
            $previousTotal = $studentData['annual_grand_total'];

            // Pass/fail based on annual percentage
            $passThreshold = settings('annual_pass_percentage', 50);
            $studentData['remark'] = $studentData['annual_average'] >= $passThreshold ? 'Pass' : 'Fail';
        }

        return $masterList;
    }

    private function calculateGrade($score, $session)
    {
        if ($score >= $session->grade_a_min) return 'A';
        if ($score >= $session->grade_b_min) return 'B';
        if ($score >= $session->grade_c_min) return 'C';
        if ($score >= $session->grade_d_min) return 'D';
        if ($score >= $session->grade_e_min) return 'E';
        return 'F';
    }

    private function getPositionSuffix($position)
    {
        $lastDigit = $position % 10;
        $lastTwoDigits = $position % 100;

        if ($lastTwoDigits >= 11 && $lastTwoDigits <= 13) {
            return $position . 'th';
        }

        switch ($lastDigit) {
            case 1:
                return $position . 'st';
            case 2:
                return $position . 'nd';
            case 3:
                return $position . 'rd';
            default:
                return $position . 'th';
        }
    }

    public function getClassesForUser($user)
    {
        if ($user->isAdmin() || $user->isHeadTeacher()) {
            return SchoolClass::where('is_active', true)->orderBy('order')->get();
        }

        if ($user->isTeacher()) {
            return $user->classes()->where('is_active', true)->orderBy('order')->get();
        }

        return collect();
    }

    public function getStudentAnnualResult($studentId, $year)
    {
        $student = User::with(['batch.schoolClass'])->find($studentId);
        $sessions = $this->getSessionsForYear($year);

        if (!$student || $sessions->isEmpty()) {
            return null;
        }

        // Check for annual approval with class snapshot
        $annualApproval = \App\Models\AnnualResultApproval::where('student_id', $studentId)
            ->where('year', $year)
            ->first();
        
        if ($annualApproval && $annualApproval->class_snapshot) {
            $class = (object) $annualApproval->class_snapshot;
        } else {
            $class = $student->batch->schoolClass;
        }

        // Try to get subjects from approval snapshots first
        $subjects = $this->getSubjectsFromApprovals(collect([$student]), $sessions->pluck('id'));
        if (!$subjects) {
            $subjects = $this->getSubjectsByClass($class->id);
        }

        $results = Result::with(['subject', 'session'])
            ->where('student_id', $studentId)
            ->whereIn('session_id', $sessions->pluck('id'))
            ->where('approved', true)
            ->get()
            ->groupBy('session_id');

        $studentData = $this->calculateStudentAnnualData($student, $subjects, $results, $sessions);

        return [
            'student' => $student,
            'year' => $year,
            'class' => $class,
            'subjects' => $subjects,
            'sessions' => $sessions,
            'studentData' => $studentData,
            'teachers' => $class->teachers,
            'generated_at' => now()
        ];
    }

    public function getStudentAnnualResultForPrint($studentId, $year)
    {
        $data = $this->getStudentAnnualResult($studentId, $year);

        if (!$data) {
            return null;
        }

        // Get annual approval data
        $approval = \App\Models\AnnualResultApproval::where('student_id', $studentId)
            ->where('year', $year)
            ->first();

        $data['approval'] = $approval;

        // Calculate overall pass/fail based on annual average
        $maxPossibleTotal = $data['subjects']->count() * 100;
        $passThreshold = $maxPossibleTotal * 0.5; // Using 50% as default
        $data['studentData']['remark'] = $data['studentData']['annual_grand_total'] >= $passThreshold ? 'Pass' : 'Fail';

        return $data;
    }

    private function calculateStudentAnnualData($student, $subjects, $results, $sessions)
    {
        $studentData = [
            'student' => $student,
            'sessions' => [],
            'annual_grand_total' => 0,
            'annual_average' => 0,
            'annual_grades' => ['A' => 0, 'B' => 0, 'C' => 0, 'D' => 0, 'E' => 0, 'F' => 0],
            'session_count' => 0
        ];

        foreach ($sessions as $session) {
            $sessionResults = $results->get($session->id, collect());
            $sessionData = [
                'session' => $session,
                'subjects' => [],
                'grand_total' => 0,
                'average' => 0,
                'grades' => ['A' => 0, 'B' => 0, 'C' => 0, 'D' => 0, 'E' => 0, 'F' => 0]
            ];

            $subjectsOfferedInSession = 0;

            foreach ($subjects as $subject) {
                $result = $sessionResults->where('subject_id', $subject->id)->first();

                // Check if subject was offered in this session
                $approval = \App\Models\ResultApproval::where('student_id', $student->id)
                    ->where('session_id', $session->id)
                    ->whereNotNull('subjects_snapshot')
                    ->first();

                $subjectOffered = false;
                if ($approval && $approval->subjects_snapshot) {
                    foreach ($approval->subjects_snapshot as $subjectData) {
                        if ($subjectData['id'] == $subject->id) {
                            $subjectOffered = true;
                            break;
                        }
                    }
                }

                if (!$subjectOffered) {
                    // Subject not offered in this session
                    $sessionData['subjects'][$subject->id] = [
                        'ca_score' => '-',
                        'exam_score' => '-',
                        'total' => '-',
                        'grade' => '-'
                    ];
                } elseif ($result) {
                    // Subject offered and student has result
                    $total = $result->total_score;
                    $grade = $this->calculateGrade($total, $session);

                    $sessionData['subjects'][$subject->id] = [
                        'ca_score' => $result->ca_score,
                        'exam_score' => $result->exam_score,
                        'total' => $total,
                        'grade' => $grade
                    ];

                    $sessionData['grand_total'] += $total;
                    $sessionData['grades'][$grade]++;
                    $studentData['annual_grades'][$grade]++;
                    $subjectsOfferedInSession++;
                } else {
                    // Subject offered but no result (student failed/absent)
                    $sessionData['subjects'][$subject->id] = [
                        'ca_score' => 0,
                        'exam_score' => 0,
                        'total' => 0,
                        'grade' => 'F'
                    ];
                    $sessionData['grades']['F']++;
                    $studentData['annual_grades']['F']++;
                    $subjectsOfferedInSession++;
                }
            }

            $sessionData['average'] = $subjectsOfferedInSession > 0 ?
                round($sessionData['grand_total'] / $subjectsOfferedInSession, 2) : 0;

            if ($sessionData['grand_total'] > 0) {
                $studentData['annual_grand_total'] += $sessionData['grand_total'];
                $studentData['session_count']++;
            }

            $studentData['sessions'][$session->id] = $sessionData;
        }

        // Calculate total possible score and annual average as percentage
        $totalPossibleScore = $subjects->count() * 100 * $sessions->count();
        $studentData['total_possible_score'] = $totalPossibleScore;
        $studentData['annual_average'] = $totalPossibleScore > 0 ?
            round(($studentData['annual_grand_total'] / $totalPossibleScore) * 100, 2) : 0;

        // Calculate total average: sum of subject averages across all sessions
        $subjectTotalAverages = 0;
        foreach ($subjects as $subject) {
            $subjectTotal = 0;
            foreach ($studentData['sessions'] as $sessionData) {
                $total = $sessionData['subjects'][$subject->id]['total'] ?? 0;
                $subjectTotal += ($total === '-' || $total === 'N/A') ? 0 : $total;
            }
            $subjectAverage = $sessions->count() > 0 ? $subjectTotal / $sessions->count() : 0;
            $subjectTotalAverages += $subjectAverage;
        }
        $studentData['total_average'] = round($subjectTotalAverages, 2);

        return $studentData;
    }

    public function getAvailableYears()
    {
        $sessions = SchoolSession::selectRaw('MIN(start_year) as min_year, MAX(end_year) as max_year')->first();

        if (!$sessions->min_year || !$sessions->max_year) {
            return collect();
        }

        $years = [];
        for ($year = $sessions->min_year; $year <= $sessions->max_year; $year++) {
            $years[] = $year;
        }

        return collect($years)->reverse();
    }

    public function formatResultValue($value)
    {
        return $value === '-' ? '-' : ($value ?? 0);
    }

    public function calculateSubjectSummary($subject, $sessions, $studentData)
    {
        $subjectTotal = 0;
        $sessionCount = 0;
        $neverOffered = true;

        foreach ($sessions as $session) {
            $sessionResult = $studentData['sessions'][$session->id]['subjects'][$subject->id] ?? null;
            $total = $sessionResult['total'] ?? 0;

            if ($total !== '-') {
                $subjectTotal += $total;
                if ($total > 0) $sessionCount++;
                $neverOffered = false;
            }
        }

        $subjectAverage = $sessionCount > 0 ? round($subjectTotal / $sessionCount, 1) : 0;
        $grade = $subjectAverage >= 70 ? 'A' : ($subjectAverage >= 60 ? 'B' : ($subjectAverage >= 50 ? 'C' : ($subjectAverage >= 45 ? 'D' : ($subjectAverage >= 40 ? 'E' : 'F'))));
        $remark = $subjectAverage >= 70 ? 'Excellent' : ($subjectAverage >= 60 ? 'Very Good' : ($subjectAverage >= 50 ? 'Good' : ($subjectAverage >= 45 ? 'Average' : ($subjectAverage >= 40 ? 'Poor' : 'Fail'))));

        return [
            'total' => $neverOffered ? '-' : $subjectTotal,
            'average' => $neverOffered ? '-' : $subjectAverage,
            'grade' => $neverOffered ? '-' : $grade,
            'remark' => $neverOffered ? '-' : $remark,
            'never_offered' => $neverOffered
        ];
    }

    // Helper functions for calculations
    public function calculateSessionMaxTotal($sessionId, $subjects, $studentData)
    {
        $subjectsOfferedInSession = 0;
        foreach ($subjects as $subject) {
            $result = $studentData['sessions'][$sessionId]['subjects'][$subject->id] ?? null;
            if ($result && $result['total'] !== '-') {
                $subjectsOfferedInSession++;
            }
        }
        return $studentData['sessions'][$sessionId]['total_possible_score'] ?? ($subjectsOfferedInSession * 100);
    }

    public function calculateAnnualMaxTotal($sessions, $subjects, $studentData)
    {
        $annualMaxTotal = 0;
        foreach ($sessions as $session) {
            $annualMaxTotal += $this->calculateSessionMaxTotal($session->id, $subjects, $studentData);
        }
        return $annualMaxTotal;
    }

    public function calculateCorrectedAnnualAverage($annualGrandTotal, $annualMaxTotal)
    {
        return $annualMaxTotal > 0 ? round(($annualGrandTotal / $annualMaxTotal) * 100, 2) : 0;
    }

    public function getPromotionStatus($correctedAnnualAverage)
    {
        $passPercentage = settings('annual_pass_percentage', 50);
        return [
            'comment' => $correctedAnnualAverage >= $passPercentage ? 'Promoted' : 'Demoted',
            'remark' => $correctedAnnualAverage >= $passPercentage ? 'Pass' : 'Fail',
            'pass_percentage' => $passPercentage,
            'is_promoted' => $correctedAnnualAverage >= $passPercentage
        ];
    }

    public function approveAnnualResults($studentIds, $year, $approvalData, $approvedBy)
    {
        foreach ($studentIds as $studentId) {
            $student = User::find($studentId);
            if (!$student || !$student->batch) continue;
            
            // Create class snapshot
            $class = $student->batch->schoolClass;
            $classSnapshot = [
                'id' => $class->id,
                'name' => $class->name,
                'alias' => $class->alias,
                'display_name' => $class->alias ?: $class->name,
            ];
            
            // Check for soft-deleted approval first
            $existingApproval = \App\Models\AnnualResultApproval::withTrashed()
                ->where('student_id', $studentId)
                ->where('year', $year)
                ->first();

            if ($existingApproval && $existingApproval->trashed()) {
                // Restore and update the soft-deleted approval
                $existingApproval->restore();
                $existingApproval->update(array_merge($approvalData, [
                    'approved_by' => $approvedBy,
                    'approved_at' => now(),
                    'class_snapshot' => $classSnapshot
                ]));
            } else {
                // Create new or update existing approval
                \App\Models\AnnualResultApproval::updateOrCreate(
                    [
                        'student_id' => $studentId,
                        'year' => $year
                    ],
                    array_merge($approvalData, [
                        'approved_by' => $approvedBy,
                        'approved_at' => now(),
                        'class_snapshot' => $classSnapshot
                    ])
                );
            }
        }
    }

    public function getStudentAnnualData($studentId, $year)
    {
        $student = User::with(['batch.schoolClass'])->find($studentId);
        $sessions = $this->getSessionsForYear($year);

        if (!$student || $sessions->isEmpty()) {
            return null;
        }

        // Check for annual approval with class snapshot
        $annualApproval = \App\Models\AnnualResultApproval::where('student_id', $studentId)
            ->where('year', $year)
            ->first();
        
        if ($annualApproval && $annualApproval->class_snapshot) {
            $class = (object) $annualApproval->class_snapshot;
        } else {
            $class = $student->batch->schoolClass;
        }

        // Try to get subjects from approval snapshots first
        $subjects = $this->getSubjectsFromApprovals(collect([$student]), $sessions->pluck('id'));
        if (!$subjects) {
            $subjects = $this->getSubjectsByClass($class->id);
        }

        $results = Result::with(['subject', 'session'])
            ->where('student_id', $studentId)
            ->whereIn('session_id', $sessions->pluck('id'))
            ->where('approved', true)
            ->get()
            ->groupBy('session_id');

        $annualData = [];
        $annualTotal = 0;
        $sessionTotals = [];

        foreach ($subjects as $subject) {
            $subjectData = [
                'name' => $subject->display_name,
                'sessions' => [],
                'annual_total' => 0,
                'annual_average' => 0
            ];

            foreach ($sessions as $session) {
                $sessionResults = $results->get($session->id, collect());
                $result = $sessionResults->where('subject_id', $subject->id)->first();

                if ($result) {
                    $subjectData['sessions'][] = [
                        'session' => $session,
                        'total' => $result->total_score
                    ];
                    $subjectData['annual_total'] += $result->total_score;

                    if (!isset($sessionTotals[$session->id])) {
                        $sessionTotals[$session->id] = 0;
                    }
                    $sessionTotals[$session->id] += $result->total_score;
                } else {
                    $subjectData['sessions'][] = [
                        'session' => $session,
                        'total' => null
                    ];
                }
            }

            $subjectData['annual_average'] = $sessions->count() > 0 ?
                round($subjectData['annual_total'] / $sessions->count(), 1) : 0;

            $annualTotal += $subjectData['annual_total'];
            $annualData[] = $subjectData;
        }

        $annualAverage = $subjects->count() > 0 && $sessions->count() > 0 ?
            round($annualTotal / ($subjects->count() * $sessions->count()), 1) : 0;

        return [
            'sessions' => $sessions,
            'subjects' => $annualData,
            'annual_total' => $annualTotal,
            'annual_average' => $annualAverage,
            'sessions_totals' => $sessionTotals
        ];
    }

    // Additional helper for getting all calculation data at once
    public function getCalculationHelpers($sessions, $subjects, $studentData)
    {
        $annualMaxTotal = $this->calculateAnnualMaxTotal($sessions, $subjects, $studentData);
        $correctedAnnualAverage = $this->calculateCorrectedAnnualAverage($studentData['annual_grand_total'], $annualMaxTotal);
        $promotionStatus = $this->getPromotionStatus($correctedAnnualAverage);

        return [
            'annual_max_total' => $annualMaxTotal,
            'corrected_annual_average' => $correctedAnnualAverage,
            'promotion_status' => $promotionStatus
        ];
    }

    private function getSchoolInfo()
    {
        return [
            'name' => Setting::get('school_name', config('app.name')),
            'address' => Setting::get('school_address', ''),
            'phone_1' => Setting::get('contact_phone_1', ''),
            'phone_2' => Setting::get('contact_phone_2', ''),
            'website' => Setting::get('school_website', ''),
            'email' => Setting::get('school_email', '')
        ];
    }
}
