<?php

namespace App\Services;

use App\Models\Result;
use App\Models\ResultApproval;
use App\Models\User;
use App\Models\SchoolClass;
use App\Models\SchoolSession;
use App\Models\Subject;
use App\Models\SubjectAlias;
use App\Models\Setting;
use Illuminate\Support\Collection;

class MasterListService
{
    public function getMasterListData($sessionId, $classId, $batchId = null)
    {
        $session = SchoolSession::find($sessionId);
        $class = SchoolClass::find($classId);
        $batch = $batchId ? \App\Models\Batch::find($batchId) : null;
        
        if (!$session || !$class) {
            return null;
        }

        $students = $batchId ? $this->getStudentsByBatch($batchId) : $this->getStudentsByClass($classId);
        
        // Get subjects and pass percentage from approval snapshots if available
        $subjectsAndPassData = $this->getSubjectsFromApprovals($students, $sessionId, $classId);
        $subjects = $subjectsAndPassData['subjects'];
        $passPercentage = $subjectsAndPassData['pass_percentage'];
        
        $results = $batchId ? $this->getResultsForBatch($sessionId, $batchId) : $this->getResultsForClass($sessionId, $classId);
        
        $masterListData = $this->calculateMasterListData($students, $subjects, $results, $session, $passPercentage);
        
        return [
            'session' => $session,
            'class' => $class,
            'batch' => $batch,
            'subjects' => $subjects,
            'masterListData' => $masterListData,
            'pass_percentage' => $passPercentage,
            'school_info' => $this->getSchoolInfo()
        ];
    }

    private function getStudentsByClass($classId)
    {
        return User::whereHas('batch', function ($query) use ($classId) {
            $query->where('school_class_id', $classId)
                  ->where('graduated', false);
        })->where('active', true)->orderBy('firstname')->get();
    }
    
    private function getStudentsByBatch($batchId)
    {
        return User::where('batch_id', $batchId)
            ->where('active', true)
            ->orderBy('firstname')
            ->get();
    }

    private function getSubjectsByClass($classId)
    {
        $class = SchoolClass::with(['subjects', 'subjectAliases'])->find($classId);
        
        return $class->subjects->map(function ($subject) use ($class) {
            $alias = $class->subjectAliases->where('subject_id', $subject->id)->first();
            $subject->display_name = $alias ? $alias->alias : $subject->name;
            return $subject;
        });
    }

    private function getResultsForClass($sessionId, $classId)
    {
        $studentIds = $this->getStudentsByClass($classId)->pluck('id');
        
        return Result::with(['student', 'subject'])
            ->where('session_id', $sessionId)
            ->whereIn('student_id', $studentIds)
            ->where('approved', true)
            ->get()
            ->groupBy('student_id');
    }
    
    private function getResultsForBatch($sessionId, $batchId)
    {
        $studentIds = $this->getStudentsByBatch($batchId)->pluck('id');
        
        return Result::with(['student', 'subject'])
            ->where('session_id', $sessionId)
            ->whereIn('student_id', $studentIds)
            ->where('approved', true)
            ->get()
            ->groupBy('student_id');
    }

    private function calculateMasterListData($students, $subjects, $results, $session, $passPercentage)
    {
        $masterList = [];
        
        foreach ($students as $student) {
            $studentResults = $results->get($student->id, collect());
            $studentData = [
                'student' => $student,
                'subjects' => [],
                'grand_total' => 0,
                'average' => 0,
                'grades' => ['A' => 0, 'B' => 0, 'C' => 0, 'D' => 0, 'E' => 0, 'F' => 0],
                'position' => 0,
                'remark' => 'Fail'
            ];

            foreach ($subjects as $subject) {
                $result = $studentResults->where('subject_id', $subject->id)->first();
                
                if ($result) {
                    $total = $result->total_score;
                    $grade = $this->calculateGrade($total, $session);
                    
                    $studentData['subjects'][$subject->id] = [
                        'ca_score' => $result->ca_score,
                        'exam_score' => $result->exam_score,
                        'total' => $total,
                        'grade' => $grade
                    ];
                    
                    $studentData['grand_total'] += $total;
                    $studentData['grades'][$grade]++;
                } else {
                    $studentData['subjects'][$subject->id] = [
                        'ca_score' => 0,
                        'exam_score' => 0,
                        'total' => 0,
                        'grade' => 'F'
                    ];
                    $studentData['grades']['F']++;
                }
            }

            $studentData['average'] = $subjects->count() > 0 ? 
                round($studentData['grand_total'] / $subjects->count(), 2) : 0;
            
            $masterList[] = $studentData;
        }

        // Sort by grand total (descending) for position calculation
        usort($masterList, function ($a, $b) {
            return $b['grand_total'] <=> $a['grand_total'];
        });

        // Calculate positions and remarks using snapshot pass percentage with tie handling
        $maxPossibleTotal = $subjects->count() * 100;
        $passThreshold = ($maxPossibleTotal * $passPercentage) / 100;

        $currentPosition = 1;
        $previousTotal = null;
        foreach ($masterList as $index => &$studentData) {
            if ($previousTotal !== null && $studentData['grand_total'] < $previousTotal) {
                $currentPosition = $index + 1;
            }
            $studentData['position'] = $currentPosition;
            $studentData['position_suffix'] = $this->getPositionSuffix($currentPosition);
            $previousTotal = $studentData['grand_total'];
            $studentData['remark'] = $studentData['grand_total'] >= $passThreshold ? 'Pass' : 'Fail';
        }

        return $masterList;
    }

    private function calculateGrade($score, $session)
    {
        if ($score >= $session->grade_a_min) return 'A';
        if ($score >= $session->grade_b_min) return 'B';
        if ($score >= $session->grade_c_min) return 'C';
        if ($score >= $session->grade_d_min) return 'D';
        if ($score >= $session->grade_e_min) return 'E';
        return 'F';
    }

    private function getPositionSuffix($position)
    {
        $lastDigit = $position % 10;
        $lastTwoDigits = $position % 100;

        if ($lastTwoDigits >= 11 && $lastTwoDigits <= 13) {
            return $position . 'th';
        }

        switch ($lastDigit) {
            case 1: return $position . 'st';
            case 2: return $position . 'nd';
            case 3: return $position . 'rd';
            default: return $position . 'th';
        }
    }

    public function getClassesForUser($user)
    {
        if ($user->isAdmin() || $user->isHeadTeacher()) {
            return SchoolClass::where('is_active', true)->orderBy('order')->get();
        }

        if ($user->isTeacher()) {
            return $user->classes()->where('is_active', true)->orderBy('order')->get();
        }

        return collect();
    }

    public function getActiveSessions()
    {
        return SchoolSession::orderBy('start_year', 'desc')->get();
    }
    
    private function getSubjectsFromApprovals($students, $sessionId, $classId)
    {
        // Check if any student has approval with snapshot
        $approvalWithSnapshot = null;
        foreach ($students as $student) {
            $approval = ResultApproval::where('student_id', $student->id)
                ->where('session_id', $sessionId)
                ->whereNotNull('subjects_snapshot')
                ->first();
            if ($approval) {
                $approvalWithSnapshot = $approval;
                break;
            }
        }
        
        if ($approvalWithSnapshot && $approvalWithSnapshot->subjects_snapshot) {
            // Use snapshot data
            $subjects = collect($approvalWithSnapshot->subjects_snapshot)->map(function ($subjectData) {
                return (object) [
                    'id' => $subjectData['id'],
                    'name' => $subjectData['name'],
                    'display_name' => $subjectData['display_name'],
                ];
            });
            $passPercentage = $approvalWithSnapshot->annual_pass_percentage;
        } else {
            // Use current class subjects
            $subjects = $this->getSubjectsByClass($classId);
            $session = SchoolSession::find($sessionId);
            $passPercentage = $session->overall_pass_percentage;
        }
        
        return [
            'subjects' => $subjects,
            'pass_percentage' => $passPercentage
        ];
    }
    
    private function getSchoolInfo()
    {
        return [
            'name' => Setting::get('school_name', config('app.name')),
            'address' => Setting::get('school_address', ''),
            'phone_1' => Setting::get('contact_phone_1', ''),
            'phone_2' => Setting::get('contact_phone_2', ''),
            'website' => Setting::get('school_website', ''),
            'email' => Setting::get('school_email', '')
        ];
    }
}