<?php

namespace App\Services;

use App\Models\PaymentRequest;
use App\Models\Payment;
use App\Models\User;
use App\Models\SchoolClass;
use App\Models\Batch;
use Illuminate\Support\Facades\DB;

class PaymentService
{
    public function createPaymentRequest(array $data)
    {
        return DB::transaction(function () use ($data) {
            $paymentRequest = PaymentRequest::create([
                'title' => $data['title'],
                'description' => $data['description'],
                'amount' => $data['amount'],
                'currency' => $data['currency'] ?? 'NGN',
                'target_type' => $data['target_type'],
                'student_id' => $data['target_type'] === 'student' ? $data['target_id'] : null,
                'school_class_id' => $data['target_type'] === 'class' ? $data['target_id'] : null,
                'batch_id' => $data['target_type'] === 'batch' ? $data['target_id'] : null,
                'created_by' => auth()->id(),
                'balance' => $data['amount'],
            ]);

            return $paymentRequest;
        });
    }

    public function updatePaymentStatus(PaymentRequest $paymentRequest, string $status, array $data = [])
    {
        return DB::transaction(function () use ($paymentRequest, $status, $data) {
            $updateData = ['status' => $status];

            if ($status === 'partial_payment' && isset($data['amount_paid'])) {
                $newAmountPaid = $paymentRequest->amount_paid + $data['amount_paid'];
                $updateData['amount_paid'] = $newAmountPaid;
                $updateData['balance'] = $paymentRequest->amount - $newAmountPaid;

                // Record the payment
                Payment::create([
                    'payment_request_id' => $paymentRequest->id,
                    'amount' => $data['amount_paid'],
                    'currency' => $paymentRequest->currency,
                    'notes' => $data['notes'] ?? null,
                    'recorded_by' => auth()->id(),
                    'payment_date' => now(),
                ]);

                // Update meta with payment history
                $meta = $paymentRequest->meta ?? [];
                $meta['payment_history'][] = [
                    'amount' => $data['amount_paid'],
                    'date' => now()->toDateTimeString(),
                    'recorded_by' => auth()->user()->name,
                    'notes' => $data['notes'] ?? null,
                ];
                $updateData['meta'] = $meta;

                // Check if fully paid
                if ($updateData['balance'] <= 0) {
                    $updateData['status'] = 'paid';
                    $updateData['balance'] = 0;
                }
            } elseif ($status === 'paid') {
                $remainingAmount = $paymentRequest->balance;
                $updateData['amount_paid'] = $paymentRequest->amount;
                $updateData['balance'] = 0;

                // Record the final payment
                Payment::create([
                    'payment_request_id' => $paymentRequest->id,
                    'amount' => $remainingAmount,
                    'currency' => $paymentRequest->currency,
                    'notes' => $data['notes'] ?? 'Full payment recorded',
                    'recorded_by' => auth()->id(),
                    'payment_date' => now(),
                ]);
            }

            $paymentRequest->update($updateData);
            return $paymentRequest->fresh();
        });
    }

    public function getPaymentRequestsForTarget(string $targetType, $targetId = null)
    {
        $query = PaymentRequest::with(['payments', 'createdBy']);

        switch ($targetType) {
            case 'student':
                $query->where('student_id', $targetId);
                break;
            case 'class':
                $query->where('school_class_id', $targetId);
                break;
            case 'batch':
                $query->where('batch_id', $targetId);
                break;
            case 'all_students':
                $query->where('target_type', 'all_students');
                break;
        }

        return $query->orderBy('created_at', 'desc')->get();
    }

    public function getAllPaymentRequests()
    {
        return PaymentRequest::with(['student', 'schoolClass', 'batch', 'createdBy', 'payments'])
            ->orderBy('created_at', 'desc')
            ->paginate(20);
    }
}