<?php

namespace App\Services;

use App\Models\User;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;

class StudentService
{
    public function createStudent(array $data)
    {
        // Generate email if not provided
        if (empty($data['email'])) {
            $data['email'] = $this->generateEmail($data['firstname'], $data['surname']);
        }

        // Generate student ID if not provided
        if (empty($data['student_id'])) {
            $data['student_id'] = $this->generateStudentId();
        } else {
            // Validate student ID is 10 digits and unique
            if (strlen($data['student_id']) !== 10 || !is_numeric($data['student_id'])) {
                throw new \InvalidArgumentException('Student ID must be exactly 10 digits');
            }

            if (User::where('student_id', $data['student_id'])->exists()) {
                throw new \InvalidArgumentException('Student ID already exists');
            }
        }

        // Set password to last 4 digits of student ID
        $data['password'] = Hash::make(substr($data['student_id'], -4));

        return User::create($data);
    }

    public function updateStudent(User $student, array $data)
    {
        // If student_id is being updated, validate it
        if (isset($data['student_id']) && $data['student_id'] !== $student->student_id) {
            if (strlen($data['student_id']) !== 10 || !is_numeric($data['student_id'])) {
                throw new \InvalidArgumentException('Student ID must be exactly 10 digits');
            }

            if (User::where('student_id', $data['student_id'])->where('id', '!=', $student->id)->exists()) {
                throw new \InvalidArgumentException('Student ID already exists');
            }

            // Update password to match new student ID
            $data['password'] = Hash::make(substr($data['student_id'], -4));
        }

        $student->update($data);
        return $student;
    }

    public function deleteStudent(User $student)
    {
        return $student->delete();
    }

    public function getStudents($perPage = 15)
    {
        return User::whereDoesntHave('roles')
            ->orderBy('firstname')
            ->paginate($perPage);
    }

    private function generateEmail($firstname, $surname)
    {
        $cleanFirstname = preg_replace('/[^a-zA-Z]/', '', $firstname);
        $cleanSurname = preg_replace('/[^a-zA-Z]/', '', $surname);
        $baseEmail = Str::lower($cleanFirstname . '.' . $cleanSurname);
        $randomNumbers = rand(1000, 9999);
        return $baseEmail . $randomNumbers . '@' . config('app.base_email_domain');
    }

    private function generateStudentId()
    {
        do {
            $studentId = str_pad(rand(1000000000, 9999999999), 10, '0', STR_PAD_LEFT);
        } while (User::where('student_id', $studentId)->exists());

        return $studentId;
    }
}
